<?php
declare(strict_types=1);

namespace App\Controller;

use Cake\Http\Session;

class CartsController extends AppController
{
    public function initialize(): void
    {
        parent::initialize();
        $this->loadComponent('Flash');
    }

    public function add()
    {
        $session = $this->request->getSession();
        $cart = $session->read('Cart') ?? ['items' => [], 'total' => 0];

        $id = $this->request->getData('product_id');
        $name = $this->request->getData('name');
        $price = (float) $this->request->getData('price');
        $imageUrl = $this->request->getData('image_url');
        $quantity = (int) $this->request->getData('quantity');

        if ($quantity <= 0) {
            $quantity = 1;
        }

        $items = $cart['items'] ?? [];

        if (isset($items[$id])) {
            $items[$id]['quantity'] += $quantity;
        } else {
            $items[$id] = [
                'product_id' => $id,
                'name' => $name,
                'price' => $price,
                'image_url' => $imageUrl,
                'quantity' => $quantity
            ];
        }

        $total = 0;
        foreach ($items as $item) {
            $price = isset($item['price']) ? (float)$item['price'] : 0;
            $qty = isset($item['quantity']) ? (int)$item['quantity'] : 0;
            $total += $price * $qty;
        }

        $session->write('Cart', [
            'items' => $items,
            'total' => $total
        ]);

        $this->Flash->success(__('Product added to cart.'));
        return $this->redirect($this->referer());
    }

    public function removeItem($productId)
    {
        $this->request->allowMethod(['post']);

        $session = $this->request->getSession();
        $cart = $session->read('Cart') ?? [];

        $items = $cart['items'] ?? [];

        if (isset($items[$productId])) {
            unset($items[$productId]);
        }

        $total = 0;
        foreach ($items as $item) {
            $price = isset($item['price']) ? (float)$item['price'] : 0;
            $qty = isset($item['quantity']) ? (int)$item['quantity'] : 0;
            $total += $price * $qty;
        }

        $session->write('Cart', [
            'items' => $items,
            'total' => $total,
        ]);

        $this->Flash->success(__('Item removed from cart.'));
        return $this->redirect(['action' => 'view']);
    }

    public function view()
    {
        $cartData = $this->request->getSession()->read('Cart') ?? [];

        $cart = $cartData['items'] ?? [];
        $total = $cartData['total'] ?? 0;

        $groupedCart = [];
        foreach ($cart as $item) {
            if (!isset($item['product_id']) || !isset($item['quantity'])) {
                continue;
            }

            $key = $item['product_id'];

            if (!isset($groupedCart[$key])) {
                $groupedCart[$key] = $item;
            } else {
                $groupedCart[$key]['quantity'] += $item['quantity'];
            }
        }

        $this->set(compact('cart', 'groupedCart', 'total'));
    }

    public function clear()
    {
        $this->request->getSession()->delete('Cart');
        $this->Flash->success(__('Cart cleared.'));
        return $this->redirect(['action' => 'view']);
    }

    // Checkout action
    public function checkout()
    {
        $session = $this->request->getSession();
        $cart = $session->read('Cart') ?? [];

        if (empty($cart)) {
            $this->Flash->error(__('Your cart is empty. Please add items before checking out.'));
            return $this->redirect(['action' => 'view']);
        }

        $session->delete('Cart');

        // Display success message
        $this->Flash->success(__('Your checkout was successful! Thank you for your purchase.'));

        // Redirect to a confirmation or other relevant page
        return $this->redirect(['controller' => 'Carts', 'action' => 'view']);
    }
}
