<?php
declare(strict_types=1);

namespace App\Controller;

use Cake\Utility\Text;
use Laminas\Diactoros\UploadedFile;

/**
 * Service Controller
 *
 * @property \App\Model\Table\ServiceTable $Service
 */
class ServiceController extends AppController
{
    public function index()
    {
        $query = $this->Service->find();
        $service = $this->paginate($query);

        $this->set(compact('service'));
    }

    private function validateId($id)
    {
        if (!ctype_digit((string)$id)) {
            return false;
        }
        return true;
    }

    public function view($id = null)
    {
        if (!$this->validateId($id)) {
            return $this->redirect(['controller' => 'Pages', 'action' => 'error']);
        }

        $service = $this->Service->find()
            ->where(['id' => $id])
            ->first();

        if (!$service) {
            return $this->redirect(['controller' => 'Pages', 'action' => 'error']);
        }

        $this->set(compact('service'));
    }

    public function add()
    {
        $service = $this->Service->newEmptyEntity();

        if ($this->request->is('post')) {
            $data = $this->request->getData();
            $image = $data['image_file'] ?? null;

            if ($image instanceof UploadedFile && $image->getError() === UPLOAD_ERR_OK) {
                $filename = Text::uuid() . '-' . $image->getClientFilename();
                $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));

                if (in_array($ext, ['jpg', 'jpeg', 'png'])) {
                    $uploadPath = WWW_ROOT . 'img/services/';
                    if (!is_dir($uploadPath)) {
                        mkdir($uploadPath, 0775, true);
                    }
                    $image->moveTo($uploadPath . $filename);
                    $data['image_url'] = 'img/services/' . $filename;
                } else {
                    $this->Flash->error(__('Invalid file type. Only JPG and PNG allowed.'));
                    return $this->redirect($this->referer());
                }
            } else {
                $this->Flash->error(__('No image was uploaded or upload failed.'));
            }

            $service = $this->Service->patchEntity($service, $data);

            if ($this->Service->save($service)) {
                $this->Flash->success(__('The service has been saved.'));
                return $this->redirect(['action' => 'index']);
            }

            $this->Flash->error(__('The service could not be saved. Please try again.'));
        }

        $this->set(compact('service'));
    }

    public function edit($id = null)
    {
        if (!$this->validateId($id)) {
            return $this->redirect(['controller' => 'Pages', 'action' => 'error']);
        }

        $service = $this->Service->find()
            ->where(['id' => $id])
            ->first();

        if (!$service) {
            return $this->redirect(['controller' => 'Pages', 'action' => 'error']);
        }

        if ($this->request->is(['patch', 'post', 'put'])) {
            $service = $this->Service->patchEntity($service, $this->request->getData());
            if ($this->Service->save($service)) {
                $this->Flash->success(__('The service has been saved.'));
                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('The service could not be saved. Please, try again.'));
        }

        $this->set(compact('service'));
    }

    public function delete($id = null)
    {
        if (!$this->validateId($id)) {
            return $this->redirect(['controller' => 'Pages', 'action' => 'error']);
        }

        $this->request->allowMethod(['post', 'delete']);

        $service = $this->Service->find()
            ->where(['id' => $id])
            ->first();

        if (!$service) {
            return $this->redirect(['controller' => 'Pages', 'action' => 'error']);
        }

        if ($this->Service->delete($service)) {
            $this->Flash->success(__('The service has been deleted.'));
        } else {
            $this->Flash->error(__('The service could not be deleted. Please, try again.'));
        }

        return $this->redirect(['action' => 'index']);
    }
}
