<?php
declare(strict_types=1);

namespace App\Controller;
use App\Model\Table\PurchasesTable;
use App\Model\Table\ProductTable;
use Cake\Routing\Router;

/**
 * Payments Controller
 *
 * @property \App\Model\Table\PaymentsTable $Payments
 */
class PaymentsController extends AppController
{
     /**
     * @var \App\Model\Table\PurchasesTable
     */
    public $Purchases;
    public $Product;

    /**
     * Index method
     *
     * @return \Cake\Http\Response|null|void Renders view
     */
    public function index()
    {
        $query = $this->Payments->find()
            ->contain(['Purchases']);
        $payments = $this->paginate($query);

        $this->set(compact('payments'));
    }

    /**
     * View method
     *
     * @param string|null $id Payment id.
     * @return \Cake\Http\Response|null|void Renders view
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function view($id = null)
    {
        $payment = $this->Payments->get($id, contain: ['Purchases']);
        $this->set(compact('payment'));
    }

    /**
     * Add method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */
    public function add()
    {
        $payment = $this->Payments->newEmptyEntity();
        if ($this->request->is('post')) {
            $payment = $this->Payments->patchEntity($payment, $this->request->getData());
            if ($this->Payments->save($payment)) {
                $this->Flash->success(__('The payment has been saved.'));

                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('The payment could not be saved. Please, try again.'));
        }
        $purchases = $this->Payments->Purchases->find('list', limit: 200)->all();
        $this->set(compact('payment', 'purchases'));
    }

    /**
     * Edit method
     *
     * @param string|null $id Payment id.
     * @return \Cake\Http\Response|null|void Redirects on successful edit, renders view otherwise.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function edit($id = null)
    {
        $payment = $this->Payments->get($id, contain: []);
        if ($this->request->is(['patch', 'post', 'put'])) {
            $payment = $this->Payments->patchEntity($payment, $this->request->getData());
            if ($this->Payments->save($payment)) {
                $this->Flash->success(__('The payment has been saved.'));

                return $this->redirect(['action' => 'index']);
            }
            $this->Flash->error(__('The payment could not be saved. Please, try again.'));
        }
        $purchases = $this->Payments->Purchases->find('list', limit: 200)->all();
        $this->set(compact('payment', 'purchases'));
    }

    /**
     * Delete method
     *
     * @param string|null $id Payment id.
     * @return \Cake\Http\Response|null Redirects to index.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function delete($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);
        $payment = $this->Payments->get($id);
        if ($this->Payments->delete($payment)) {
            $this->Flash->success(__('The payment has been deleted.'));
        } else {
            $this->Flash->error(__('The payment could not be deleted. Please, try again.'));
        }

        return $this->redirect(['action' => 'index']);
    }
    
    public function success()
    {
        
        $session = $this->request->getSession();

        if ($session->read('PaymentProcessed')) {
            return $this->redirect('/');
        }
        
        $this->Purchases = $this->fetchTable('Purchases');
        $this->Product = $this->fetchTable('Product');
        
        $purchaseId = $this->request->getQuery('purchase_id');
        $totalPrice = $this->request->getQuery('total_price');
        $this->set('message', 'Your payment was successful.');
        
        $payment = $this->Payments->newEmptyEntity();
        $payment->purchase_id = $purchaseId;
        $payment->total_price = $totalPrice;
        $payment->method = 'card';
        $payment->state = 'success';
        $payment->paid_on = date('Y-m-d H:i:s');
    
         if ($this->Payments->save($payment)) {
            
        
            $id = $this->request->getQuery('purchase_id');

            if (!$id || !is_numeric($id)) {
                throw new NotFoundException(__('Invalid purchase ID.'));
            }

            $purchase = $this->Purchases->get((int)$id, [
                'contain' => ['ProductPurchase']
            ]);
            $purchase->status = 'paid';
            $purchase->delivered = 'Not Delivered';
            $this->Purchases->save($purchase);
            
            foreach ($purchase->product_purchase as $link) {
                $product = $this->Product->get($link->product_id);
                $product->stock -= $link->quantity;
                if ($product->stock < 0) {
                    $product->stock = 0;
                }
                $this->Product->save($product);
            }
            

        } else {
            $this->Flash->error('Payment failed.');
            return $this->redirect('/');
        }
        
        $this->getRequest()->getSession()->delete('Cart');
        $this->set(compact('payment'));
        
        $session->write('PaymentProcessed', true);
        return $this->redirect(['action' => 'end']);

    }

    public function cancel()
    {
        $purchaseId = $this->request->getQuery('purchase_id');

        if (!$purchaseId || !is_numeric($purchaseId)) {
            return $this->redirect('/');
        }

        $this->Purchases = $this->fetchTable('Purchases');

        $purchase = $this->Purchases->get((int)$purchaseId);
        $purchase->status = 'cancelled';
        $this->Purchases->save($purchase);

        $message = 'You returned from checkout. This purchase has been cancelled.';
        $redirectUrl = Router::url(['controller' => 'Carts', 'action' => 'view'], true);

        echo "<script>
            alert(" . json_encode($message) . ");
            window.location.href = '$redirectUrl';
        </script>";

        return $this->response->withType('text/html');
    }
    
    public function end()
    {
        $this->request->getSession()->delete('PaymentProcessed');
    }
    
}
