<?php
declare(strict_types=1);

namespace App\Controller;

use Cake\Core\Configure;
use Cake\Http\Exception\BadRequestException;
use Stripe\Stripe;
use Stripe\Checkout\Session;
use Cake\Datasource\ConnectionManager;

class StripeController extends AppController
{
    public function initialize(): void
    {
        parent::initialize();
        Stripe::setApiKey(Configure::read('Stripe.secret'));
    }

    public function checkout()
    {
        $this->viewBuilder()->disableAutoLayout();
        $this->set('publishableKey', Configure::read('Stripe.publishable'));
    }

    public function createSession()
    {
        $this->request->allowMethod(['post']);

        try {
            $session = Session::create([
                'payment_method_types' => ['card'],
                'line_items' => [[
                    'price_data' => [
                        'currency' => 'usd',
                        'product_data' => [
                            'name' => 'Sample Product',
                        ],
                        'unit_amount' => 2000,
                    ],
                    'quantity' => 1,
                ]],
                'mode' => 'payment',
                'success_url' => 'https://dev.u25s1019.iedev.org/stripe/success',
                'cancel_url' => 'https://dev.u25s1019.iedev.org/stripe/cancel',
            ]);

            return $this->redirect($session->url);
        } catch (\Exception $e) {
            throw new BadRequestException('Unable to create Stripe session: ' . $e->getMessage());
        }
    }
    
    public function webhook()
    {
        $this->request->allowMethod(['post']);

        $payload = $this->request->input();
        $event = json_decode($payload, true);

        if (empty($event['type']) || empty($event['data']['object']['id'])) {
            throw new BadRequestException('Invalid webhook data');
        }

        $sessionId = $event['data']['object']['id'];

        $conn = ConnectionManager::get('default');

        if ($event['type'] === 'checkout.session.expired') {
            $conn->execute(
                "UPDATE purchases SET state = 'cancelled' WHERE stripe_session_id = :session_id",
                ['session_id' => $sessionId]
            );
        }

        if ($event['type'] === 'checkout.session.completed') {
            $conn->execute(
                "UPDATE purchases SET state = 'completed' WHERE stripe_session_id = :session_id",
                ['session_id' => $sessionId]
            );

            $amountTotal = $event['data']['object']['amount_total'] ?? null;
            if ($amountTotal !== null) {
                $conn->execute(
                    "UPDATE purchases SET total_price = :total_price WHERE stripe_session_id = :session_id",
                    ['total_price' => $amountTotal / 100, 'session_id' => $sessionId]
                );
            }
        }

        return $this->response->withStringBody('Webhook received');
    }

    public function success()
    {
        // Handle post-payment success logic here
    }

    public function cancel()
    {
        // Handle payment cancellation logic here
    }

}
