<?php
/**
 * @var \App\View\AppView $this
 * @var array $faqs
 */

use Cake\Datasource\ConnectionManager;
use Cake\Core\Configure;
use Cake\Http\Exception\NotFoundException;

$this->disableAutoLayout();

$checkConnection = function (string $name) {
    $error = null;
    $connected = false;
    try {
        ConnectionManager::get($name)->getDriver()->connect();
        $connected = true;
    } catch (Exception $connectionError) {
        $error = $connectionError->getMessage();
        if (method_exists($connectionError, 'getAttributes')) {
            $attributes = $connectionError->getAttributes();
            if (isset($attributes['message'])) {
                $error .= '<br />' . $attributes['message'];
            }
        }
    }
    return compact('connected', 'error');
};

if (!Configure::read('debug')) {
    throw new NotFoundException('Page not available in production.');
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <?= $this->Html->charset() ?>
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Frequently Asked Questions</title>
    <?= $this->Html->meta('icon') ?>
    <?= $this->Html->css('home.css') ?>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Montserrat:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    
    <style>
        /* FAQ specific styles */
        .faq-section {
            padding: 60px 50px;
            background-color: #121212;
            min-height: 70vh;
        }
        
        .faq-container {
            max-width: 800px;
            margin: 0 auto;
        }
        
        .faq-title {
            font-size: 32px;
            margin-bottom: 40px;
            text-align: center;
            position: relative;
            font-weight: 700;
            letter-spacing: 1px;
        }
        
        .faq-title:after {
            content: '';
            position: absolute;
            bottom: -15px;
            left: 50%;
            transform: translateX(-50%);
            width: 80px;
            height: 3px;
            background-color: #ff4d4d;
        }
        
        .faq-item {
            background-color: #1a1a1a;
            border-radius: 8px;
            margin-bottom: 20px;
            overflow: hidden;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            transition: transform 0.3s, box-shadow 0.3s;
        }
        
        .faq-item:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 25px rgba(0,0,0,0.2);
        }
        
        .faq-question {
            padding: 20px 25px;
            font-weight: 600;
            font-size: 18px;
            cursor: pointer;
            display: flex;
            justify-content: space-between;
            align-items: center;
            transition: background-color 0.3s;
        }
        
        .faq-question:hover {
            background-color: #222;
        }
        
        .faq-question i {
            color: #ff4d4d;
            transition: transform 0.3s;
        }
        
        .faq-question.active i {
            transform: rotate(180deg);
        }
        
        .faq-answer {
            padding: 0;
            max-height: 0;
            overflow: hidden;
            transition: max-height 0.4s ease, padding 0.4s ease;
            border-top: 0px solid #2a2a2a;
        }
        
        .faq-answer.active {
            padding: 20px 25px;
            max-height: 1000px;
            border-top: 1px solid #2a2a2a;
        }
        
        .faq-text {
            color: #d0d0d0;
            line-height: 1.8;
        }
        
        /* Brand link style */
        .brand {
            font-size: 24px;
            font-weight: 700;
            font-family: 'Montserrat', sans-serif;
            transition: color 0.3s;
            padding-left: 30px;
        }
        
        .brand:hover {
            color: #ff4d4d;
        }
        
        /* Mobile menu styles */
        .mobile-menu-icon {
            display: none;
            font-size: 24px;
            cursor: pointer;
        }
        
        .mobile-menu {
            display: none;
            position: fixed;
            top: 0;
            left: -100%;
            width: 80%;
            height: 100%;
            background-color: #1f1f1f;
            z-index: 999;
            padding: 50px 30px;
            transition: left 0.3s ease;
        }
        
        .mobile-menu.active {
            left: 0;
        }
        
        .mobile-menu-close {
            position: absolute;
            top: 20px;
            right: 20px;
            font-size: 24px;
            cursor: pointer;
        }
        
        .mobile-menu a {
            display: block;
            margin: 20px 0;
            font-size: 18px;
        }
        
        /* Responsive adjustments */
        @media screen and (max-width: 991px) {
            .faq-section {
                padding: 40px 30px;
            }
        }
        
        @media screen and (max-width: 768px) {
            .navbar {
                padding: 20px;
            }
            
            .links {
                display: none;
            }
            
            .mobile-menu-icon {
                display: block;
            }
            
            .mobile-menu {
                display: block;
            }
            
            .faq-section {
                padding: 30px 20px;
            }
            
            .faq-title {
                font-size: 28px;
            }
            
            .faq-question {
                font-size: 16px;
                padding: 15px 20px;
            }
            
            .faq-answer.active {
                padding: 15px 20px;
            }
        }
    </style>
</head>
<body>
    <!-- Navigation Bar -->
    <div class="navbar">
        <div class="brand">
            <!-- Wrap the brand name with a clickable anchor tag but preserve the same style -->
            <a href="<?= $this->Url->build('/') ?>" style="text-decoration: none; color: inherit; font-family: 'Montserrat', sans-serif; font-weight: 700; font-size: 24px;">
                PowerPro Shop
            </a>
        </div>
        <div class="links">
            <a href="<?= $this->Url->build(['controller' => 'Product', 'action' => 'index']) ?>">Our Products</a>
            <a href="<?= $this->Url->build(['controller' => 'Service', 'action' => 'index']) ?>">Our Services</a>
            <a href="<?= $this->Url->build(['controller' => 'Faq', 'action' => 'faq']) ?>">FAQ</a>
            <a href="<?= $this->Url->build(['controller' => 'ContactForm', 'action' => 'add']) ?>">Contact Us</a> <!-- Desktop link -->
            <a href="<?= $this->Url->build(['controller' => 'Auth', 'action' => 'login']) ?>" class="login-btn" style="border: 1px solid #ff4d4d; padding: 8px 15px; border-radius: 4px; color: #ff4d4d;">Login</a>
            <?= $this->Html->link('<i class="fas fa-shopping-cart"></i>',
                ['controller' => 'Carts', 'action' => 'view'],
                ['class' => 'cart-link', 'escape' => false]
            ) ?>
        </div>
        <div class="mobile-menu-icon">
            <i class="fas fa-bars"></i>
        </div>
    </div>

    
    <!-- Mobile Menu -->
    <div class="mobile-menu">
        <div class="mobile-menu-close">
            <i class="fas fa-times"></i>
        </div>
        <a href="<?= $this->Url->build(['controller' => 'Product', 'action' => 'index']) ?>">Our Products</a>
        <a href="<?= $this->Url->build(['controller' => 'Service', 'action' => 'index']) ?>">Our Services</a>
        <a href="<?= $this->Url->build(['controller' => 'Faq', 'action' => 'faq']) ?>">FAQ</a>
        <a href="<?= $this->Url->build(['controller' => 'ContactForm', 'action' => 'add']) ?>">Contact Us</a> <!-- Added for mobile -->
        <a href="<?= $this->Url->build(['controller' => 'Auth', 'action' => 'login']) ?>">Login</a>
        <a href="<?= $this->Url->build(['controller' => 'Carts', 'action' => 'view']) ?>">
            <i class="fas fa-shopping-cart"></i> View Cart
        </a>
    </div>

    
    <!-- FAQ Section -->
    <div class="faq-section">
        <div class="faq-container">
            <h2 class="faq-title">Frequently Asked Questions</h2>
            
            <?php foreach ($faqs as $index => $faq): ?>
                <div class="faq-item">
                    <div class="faq-question" onclick="toggleFAQ(<?= $index ?>)">
                        <?= h($faq['question']) ?>
                        <i class="fas fa-chevron-down"></i>
                    </div>
                    <div class="faq-answer" id="faq-answer-<?= $index ?>">
                        <div class="faq-text"><?= h($faq['answer']) ?></div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
    
    <!-- Footer -->
    <div class="footer">
        <div class="footer-content">
            <!-- Column 1 -->
            <div class="footer-column">
                <h3>PowerPro Shop</h3>
                <p>Your premier destination for high-quality fitness equipment and expert services to help you achieve your fitness goals.</p>
            </div>
            
            <!-- Column 2 -->
            <div class="footer-column">
                <h3>Quick Links</h3>
                <ul>
                    <li><a href="<?= $this->Url->build('/') ?>">Home</a></li>
                    <li><a href="<?= $this->Url->build(['controller' => 'Product', 'action' => 'index']) ?>">Products</a></li>
                    <li><a href="<?= $this->Url->build(['controller' => 'Service', 'action' => 'index']) ?>">Services</a></li>
                </ul>
            </div>
            
            <!-- Column 3 -->
            <div class="footer-column">
                <h3>Payment Methods</h3>
                <ul>
                    <li><i class="fab fa-cc-visa"></i> Visa</li>
                    <li><i class="fab fa-cc-mastercard"></i> Mastercard</li>
                    <li><i class="fab fa-cc-amex"></i> American Express</li>
                    <li><i class="fab fa-cc-paypal"></i> PayPal</li>
                </ul>
            </div>
            
            <!-- Column 4 -->
            <div class="footer-column">
                <h3>Contact Info</h3>
                <ul>
                    <li><i class="fas fa-map-marker-alt"></i> 123 Fitness Street, Melbourne, Australia</li>
                    <li><i class="fas fa-phone"></i> <a href="tel:(03) 1234 5678">(03) 1234 5678</a></li>
                    <li><i class="fas fa-envelope"></i> <a href="mailto:info@powerproshop.com">info@powerproshop.com</a></li>
                    <li><i class="fas fa-clock"></i> Mon-Fri: 9am-6pm, Sat: 10am-4pm</li>
                </ul>
            </div>
        </div>
        
        <div class="copyright">
            &copy; <?= date('Y') ?> PowerPro Shop. All Rights Reserved.
        </div>
    </div>
    
    <script>
        // Mobile menu toggle
        document.querySelector('.mobile-menu-icon').addEventListener('click', function() {
            document.querySelector('.mobile-menu').classList.add('active');
        });
        
        document.querySelector('.mobile-menu-close').addEventListener('click', function() {
            document.querySelector('.mobile-menu').classList.remove('active');
        });
        
        // FAQ accordion functionality
        function toggleFAQ(index) {
            const answer = document.getElementById('faq-answer-' + index);
            const question = answer.previousElementSibling;
            
            if (answer.classList.contains('active')) {
                answer.classList.remove('active');
                question.classList.remove('active');
            } else {
                // Close all other open FAQs (optional, comment out for multiple open FAQs)
                document.querySelectorAll('.faq-answer').forEach(item => {
                    item.classList.remove('active');
                });
                document.querySelectorAll('.faq-question').forEach(item => {
                    item.classList.remove('active');
                });
                
                // Open the clicked FAQ
                answer.classList.add('active');
                question.classList.add('active');
            }
        }
    </script>
</body>
</html>
